/** @file   bullet.cpp
 * @brief   Implementation of Bullet - class.
 * @version $Revision: 1.4 $
 * @author  Tomi Lamminsaari
 */

#include "bullet.h"
#include "www_map.h"
#include "gameobject.h"
#include "weapon.h"
#include "gameanims.h"
#include "animplayer.h"
#include "www_assert.h"
#include "gfxid.h"
#include "AnimId.h"
using namespace eng2d;

namespace WeWantWar {

///
/// Static members, constants and datatypes
/// =======================================




/** Returns the typecode for the bullet that matches the given weapon.
 */
Bullet::TType Bullet::findBulletType( const Weapon& aWeapon )
{
  TType t;
  switch ( aWeapon.id() ) {
    case ( Weapon::W_RIFLE ): {
      t = ERifle;
      break;
    }
    case ( Weapon::W_SHOTGUN ): {
      t = EShotgun;
      break;
    }
    case ( Weapon::W_CROWBAR ): {
      t = ECrowbar;
      break;
    }
    case ( Weapon::W_FLAMETHROWER ): {
      t = EFlameThrower;
      break;
    }
    case ( Weapon::W_ALIENFIREBALL ): {
      t = EAlienFireball;
      break;
    }
    case ( Weapon::W_MINIGUN ): {
      t = EMinigun;
      break;
    }
    case ( Weapon::W_UZI ): {
      t = EUzi;
      break;
    }
    case ( Weapon::W_GRENADE ): {
      t = EGrenade;
      break;
    }
    case ( Weapon::W_TANKBULLET ): {
      t = ETankBullet;
      break;
    }
    case ( Weapon::W_SENTRYFIREBALL ): {
      t = ESentryFireball;
      break;
    }
    case ( Weapon::W_SNIPERRIFLE ): {
      t = ESniperRifle;
      break;
    }
    case ( Weapon::W_ROCKETLAUNCHER ): {
      t = ERocketLauncher;
      break;
    }
    default: {
      t = EInvisible;
      break;
    }
  }
  return t;
}



/** Returns an instance of weapon that shoots the bullets of given type.
 */
Weapon Bullet::bullet2Weapon( Bullet::TType aType )
{
  switch ( aType ) {
    case ( Bullet::ERifle ): {
      return Weapon( Weapon::W_RIFLE );
    }
    case ( Bullet::EShotgun ): {
      return Weapon( Weapon::W_SHOTGUN );
    }
    case ( Bullet::EGrenade ): {
      return Weapon( Weapon::W_GRENADE );
    }
    case ( Bullet::EAlienFireball ): {
      return Weapon( Weapon::W_ALIENFIREBALL );
    }
    case ( Bullet::EFlameThrower ): {
      return Weapon( Weapon::W_FLAMETHROWER );
    }
    case ( Bullet::EMinigun ): {
      return Weapon( Weapon::W_MINIGUN );
    }
    case ( Bullet::EUzi ): {
      return Weapon( Weapon::W_UZI );
    }
    case ( Bullet::ECrowbar ): {
      return Weapon( Weapon::W_CROWBAR );
    }
    case ( Bullet::ETankBullet ): {
      return Weapon( Weapon::W_TANKBULLET );
    }
    case ( Bullet::ESentryFireball ): {
      return Weapon( Weapon::W_SENTRYFIREBALL );
    }
    case ( Bullet::ESniperRifle ): {
      return Weapon( Weapon::W_SNIPERRIFLE );
    }
    case ( Bullet::ERocketLauncher ): {
      return Weapon( Weapon::W_ROCKETLAUNCHER );
    }
    default: {
      return Weapon( Weapon::W_NOWEAPON );
    }
  }
}



///
/// Constructors, destructor and operators
/// ======================================

/** Constructor
 */
Bullet::Bullet( GameObject* aShooter, TType aType, const Vec2D& aPos ) :
  Ammunition( aShooter ),
  iType( aType ),
  iOrigin( aPos ),
  iRange( 0 ),
  iDamage( 0 ),
  iRadius( 1 )
{
  int angle = 0;
  if ( aShooter != 0 ) {
    angle = aShooter->angle();
  }
  
  // Create the weapon object for ammunition info. Set up the general data.
  Weapon weapon = bullet2Weapon( aType );
  iPosition = aPos;
  iRange = weapon.range();
  iDamage = weapon.damage();
  iRadius = weapon.radius();
  iAltitude = 30;
  iRotation = angle;
    
  // Set the flags
  iFlags = 0;
  if ( weapon.bulletTrails() == true ) {
    iFlags |= Ammunition::KTrails;
  }
  if ( weapon.hitSound() == true ) {
    iFlags |= Ammunition::KMakeSound;
  }
  if ( weapon.bouncingBullets() == true ) {
    iFlags |= Ammunition::KBounce;
  }
  
  // Setup the movement vectors.
  Vec2D mvec( 0, weapon.velocity() );
  mvec.rotate( angle );
  if ( weapon.deviation() > 0 ) {
    mvec.rotate( (rand() % weapon.deviation()) - (weapon.deviation()/2) );
  }
  this->setVelocity( mvec );
}



/** Destructor
 */
Bullet::~Bullet()
{
}




///
/// Public methods
/// ==============

/** Updates this bullet
 */
bool Bullet::updateAmmunition()
{
  if ( iDone == true ) {
    return true;
  }
  
  // Manage the flying range.
  iRange -= iVelocityVectorLength;
  if ( iRange < 0 ) {
    // The bullet has been flying far enough
    iDone = true;
  }
  
  // We create the bullet trail animation
  if ( (iFlags & Ammunition::KTrails) != 0 ) {
    if ( iType == ERocketLauncher ) {
      const Animation& flameAnim = GameAnims::findAnimation( AnimId::KEffectRocketSmoke );
      AnimPlayer::spawn( flameAnim, iPosition, 0 );
    } else if ( iType == EAlienFireball2 ) {
      const Animation flameAnim = GameAnims::findAnimation( AnimId::KEffectBulletMotion2 );
      AnimPlayer::spawn( flameAnim, iPosition, 0 );
    } else {
      const Animation& motionAnim = GameAnims::findAnimation( AnimId::KEffectBulletMotion );
      AnimPlayer::spawn( motionAnim, iPosition, 0 );
    }
  }
    
  return iDone;
}



/** Draws this bullet
 */
void Bullet::redraw( BITMAP* pB ) const
{
  int x = iPosition.intX() - Map::scrollX;
  int y = iPosition.intY() - Map::scrollY;
  
  switch ( iType ) {
    case ( EUzi ):
    case ( EShotgun ):
    case ( ERifle ): {
      putpixel( pB, x,y, makecol(180,180,180) );
      break;
    }
    case ( EMinigun ): {
      hline( pB, x-1, y, x+1, makecol(230,230,230) );
      putpixel( pB, x, y-1, makecol(230,230,230) );
      putpixel( pB, x, y+1, makecol(230,230,230) );
      break;
    }
    case ( EGrenade ): {
      break;
    }
    default: {
      putpixel( pB, x,y, makecol(230,230,230) );
      break;
    }
  }
}




///
/// Getter methods
/// ==============






///
/// Private or Protected methods
/// ============================



} // end of namespace
